/**
 * \file: create_signature_db_cmd.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * \component: authorization level daemon
 *
 * \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
 *
 * \copyright (c) 2017 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/
#include "create_signature_db_cmd/create_signature_db_cmd.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/fcntl.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <limits.h>

#include "util/helper.h"
#include "util/logger.h"
#include "util/cmdline_parser.h"
#include "util/cfgfile_parser.h"

#include "control/configuration_defaults.h"
#include "model/daemon_model.h"

#include "create_signature_db_cmd/feature_db.h"
#include "create_signature_db_cmd/level_db.h"
#include "create_signature_db_cmd/signature_db_writer.h"

static cfg_item_t script_root_cfg	= {.spec = &script_root_cfg_spec, .cmdline = &script_root_cmdline_spec };
static cfg_item_t sysroot_cfg		= {.spec = &sysroot_cfg_spec, .cmdline = &sysroot_cmdline_spec };
static cfg_item_t sig_db_cfg		= {.spec = &sigdb_cfg_spec, .cmdline = &sig_db_cmdline_spec };
static cfg_item_t sigdb_privkey_cfg 	= {.spec = &sigdb_privkey_cfg_spec, .cmdline = &sigdb_privkey_cmdline_spec };
static cfg_item_t conffile_cfg 		= {.spec = &conf_file_cfg_spec, .cmdline = &conffile_cmdline_spec };
static cfg_item_t help_cfg 		= {.spec = &help_cfg_spec, .cmdline = &help_cmdline_spec };

static cfg_item_t *const create_sigdb_cfgs[] = {
		&script_root_cfg,
		&sysroot_cfg,
		&sig_db_cfg,
		&sigdb_privkey_cfg,
		&conffile_cfg,
		&help_cfg
};

static error_code_t create_signature_db_cmd_init(void);
static error_code_t create_signature_db_cmd_parse_args(const char *binary_name, int argc, char *argv[]);
static error_code_t create_signature_db_cmd_start(void);
static error_code_t create_signature_db_cmd_init(void);
static void create_signature_db_cmd_deinit(void);
static void create_signature_db_cmd_print_help(void);
static void create_signature_db_cmd_print_usage(void);

static error_code_t create_signature_db_cmd_init(void)
{
	logger_init_console_only(LOGGER_LEVEL_ERROR);
	return RESULT_OK;
}

static error_code_t create_signature_db_cmd_parse_args(const char *binary_name, int argc, char *argv[])
{
	error_code_t result = RESULT_OK;
	int create_sigdb_cfgs_cnt = ARRAY_SIZE(create_sigdb_cfgs);

	(void) binary_name;

	helper_items_init(create_sigdb_cfgs, create_sigdb_cfgs_cnt);

	result = cmdline_parser_parse_cmd_line(create_sigdb_cfgs, create_sigdb_cfgs_cnt, argv, argc);

	if (result == RESULT_INVALID_ARGS) {
		create_signature_db_cmd_print_usage();
		return RESULT_INVALID_ARGS;
	}

	if (helper_get_flag(&help_cfg))
	{
		create_signature_db_cmd_print_help();
		return RESULT_HELP_PRINTED;
	}

	if (cfgfile_parser_is_any_cfg_item_unset(create_sigdb_cfgs, create_sigdb_cfgs_cnt))
		result = cfgfile_parser_parse_config_file(helper_get_str_value(&conffile_cfg),
				create_sigdb_cfgs, create_sigdb_cfgs_cnt);

	return result;
}

static void create_signature_db_cmd_print_help(void)
{
	printf("\n");
	printf("Authorization Level Daemon Initialization Utility - Used to create configurations and\n"
			"signatures for the ADIT authorization level daemon.\n\n");
	printf("Usage: ald_init %s [OPTIONS...]\n",create_signature_db_cmd_vtable.command);
	printf("This command scans all scripts and level files, checks for consistency, and creates\n"
			"finally a signature database\n\n");
	printf("\t"SCRIPT_ROOT_CMD_SHORT", "SCRIPT_ROOT_CMD_LONG"\t\tRoot dir of the script repository\n");
	printf("\t"SYSROOT_CMD_SHORT", "SYSROOT_CMD_LONG"\t\tDestination of an offline rfs to which the db is to be stored\n");
	printf("\t"SIGDB_PRIVKEY_CMD_SHORT", "SIGDB_PRIVKEY_CMD_LONG"\tPath of the private key used to sign scripts and db\n");
	printf("\t"SIG_DB_CMD_SHORT", "SIG_DB_CMD_LONG"\t\tPath to the file where the db is stored in\n");
	printf("\t"CONFFILE_CMD_SHORT", "CONFFILE_CMD_LONG"\t\tLoad the db file path and rootdir path from this config file\n");
	printf("\t"HELP_CMD_SHORT", "HELP_CMD_LONG"\t\tDisplays this help and exits\n\n");
	printf("The scripts and level files are expected in a well defined structure below the\n"
			"given root directory set with "SCRIPT_ROOT_CMD_LONG" option. Please refer to the manual document\n"
			"for the exact structure expected below rootdir. If the rootdir is not passed,\n"
			"its taken from the config file (Default : \"%s\"). If the rootdir is \n"
			"not available in conf file, the default directory \"%s\" is used \n\n"
			"The signatures for the scripts and for the data base are created using private key\n"
			"passed with the --sigdbprivkeypath option. If no private key is given, a key named\n"
			"\""SIGDB_PRIV_KEY_NAME"\" is expected in the current directory. The created data base is\n"
			"stored under the path passed with --sigdbpath option. If sigdbpath is not passed,\n"
			"the path is taken from the config file. If the sigdbpath is not available in conf file,\n"
			"then the default location \"%s\" is used.\n\n"
			"A sysroot path can be passed using --sysroot option to create the database for an\n"
			"offline RFS on host side.\n\n",helper_get_str_default(conffile_cfg.spec),
			helper_get_str_default(script_root_cfg.spec), helper_get_str_default(sig_db_cfg.spec));
}

static void create_signature_db_cmd_print_usage(void)
{
	printf("\n");
	printf("Usage: ald_init %s [OPTIONS...]\n\n",create_signature_db_cmd_vtable.command);
	printf("\t"SCRIPT_ROOT_CMD_SHORT", "SCRIPT_ROOT_CMD_LONG"\t\tRoot dir of the script repository\n");
	printf("\t"SYSROOT_CMD_SHORT", "SYSROOT_CMD_LONG"\t\tDestination of an offline rfs to which the db is to be stored\n");
	printf("\t"SIGDB_PRIVKEY_CMD_SHORT", "SIGDB_PRIVKEY_CMD_LONG"\tPath of the private key used to sign scripts and db\n");
	printf("\t"SIG_DB_CMD_SHORT", "SIG_DB_CMD_LONG"\t\tPath to the file where the db is stored in\n");
	printf("\t"CONFFILE_CMD_SHORT", "CONFFILE_CMD_LONG"\t\tLoad the db file path and rootdir path from this config file\n");
	printf("\t"HELP_CMD_SHORT", "HELP_CMD_LONG"\t\tDisplays this help and exits\n\n");
}

static error_code_t create_signature_db_cmd_start(void)
{
	error_code_t result;
	const char *script_root_dir = helper_get_str_value(&script_root_cfg);
	const char *sysroot_dir		= helper_get_str_value(&sysroot_cfg);
	const char *signature_db_path = helper_get_str_value(&sig_db_cfg);

	//loading feature set
	printf("--------------------- Loading and checking feature set ... ---------------------------\n");
	result=feature_db_load(script_root_dir,sysroot_dir);

	//checking level conf files
	if (result==RESULT_OK)
	{
		printf("-------------------------------- [DONE] ----------------------------------------------\n\n");
		printf("------------------ Checking level for consistency ... --------------------------------\n");
		result=level_db_check_level_consistency(script_root_dir,sysroot_dir);
	}

	if (result==RESULT_OK)
	{
		printf("-------------------------------- [DONE] ----------------------------------------------\n\n");
		printf("---------------------- Creating signature db ... -------------------------------------\n");
		result=signature_db_writer_init(helper_get_str_value(&sigdb_privkey_cfg));
	}

	//creating the signature db file
	if (result==RESULT_OK)
		result=signature_db_writer_create_db(sysroot_dir, signature_db_path);

	//writing down the signature db part that belongs to the feature scripts
	if (result==RESULT_OK)
		result=feature_db_fill_signature_db(script_root_dir);

	//writing down the signature db part that belongs to the level scripts and files
	if (result==RESULT_OK)
		result=level_db_fill_signature_db(script_root_dir,sysroot_dir);

	//closing the db file
	signature_db_writer_close();

	//creating a signature of the data base
	if (result==RESULT_OK)
	{
		printf("-------------------------------- [DONE] ----------------------------------------------\n\n");
		printf("--------------------- signing the signature db ... -----------------------------------\n");
		result=signature_db_writer_sign_db(sysroot_dir, signature_db_path);
	}

	signature_db_writer_deinit();

	//a data base file might have been created but the creation didn't succeed completely -> delete the file
	if (result!=RESULT_OK)
	{
		(void)unlink(signature_db_path);
		printf("Signature db generation failed.\n");
	}
	else
	{
		printf("-------------------------------- [DONE] ----------------------------------------------\n\n");
		printf("Signature db generation successful.\n");
	}

	return result;
}

static void create_signature_db_cmd_deinit(void)
{
	feature_db_free();
	helper_items_free(create_sigdb_cfgs, ARRAY_SIZE(create_sigdb_cfgs));
}

command_vtable_t create_signature_db_cmd_vtable=
{
		.command_description="Create the signature data base the daemon uses to verify that scripts are not modified.",
		.command="create_signature_db",
		.init=create_signature_db_cmd_init,
		.parse_args=create_signature_db_cmd_parse_args,
		.start=create_signature_db_cmd_start,
		.deinit=create_signature_db_cmd_deinit
};
